"use client"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import {
  ShoppingCart,
  Search,
  User,
  Menu,
  Star,
  Heart,
  Share2,
  ChevronRight,
  Plus,
  Minus,
  Truck,
  Shield,
  RotateCcw,
} from "lucide-react"

// Mock product data - in real app this would come from API/database
const getProductById = (id: string) => {
  const products = {
    "1": {
      id: 1,
      name: "iPhone 15 Pro Max",
      price: 1199,
      originalPrice: 1299,
      images: [
        "/iphone-15-pro-max.png",
        "/placeholder.svg?height=500&width=500&text=iPhone+Side+View",
        "/placeholder.svg?height=500&width=500&text=iPhone+Back+View",
        "/placeholder.svg?height=500&width=500&text=iPhone+Camera+Detail",
      ],
      rating: 4.9,
      reviews: 2847,
      badge: "Best Seller",
      category: "Smartphones",
      brand: "Apple",
      description:
        "The iPhone 15 Pro Max features a titanium design, advanced camera system with 5x telephoto zoom, and the powerful A17 Pro chip. Experience professional photography and videography capabilities in your pocket.",
      features: [
        "6.7-inch Super Retina XDR display",
        "A17 Pro chip with 6-core GPU",
        "Pro camera system with 5x telephoto",
        "Titanium design with Ceramic Shield",
        "Up to 29 hours video playback",
        "5G connectivity",
      ],
      specifications: {
        Display: "6.7-inch Super Retina XDR OLED",
        Processor: "A17 Pro chip",
        Storage: "256GB, 512GB, 1TB",
        Camera: "48MP Main, 12MP Ultra Wide, 12MP Telephoto",
        Battery: "Up to 29 hours video playback",
        "Operating System": "iOS 17",
        Connectivity: "5G, Wi-Fi 6E, Bluetooth 5.3",
        Colors: "Natural Titanium, Blue Titanium, White Titanium, Black Titanium",
      },
      inStock: true,
      stockCount: 15,
    },
    "2": {
      id: 2,
      name: "Samsung Galaxy S24 Ultra",
      price: 1099,
      originalPrice: 1199,
      images: [
        "/samsung-galaxy-s24-ultra.png",
        "/placeholder.svg?height=500&width=500&text=Galaxy+S24+Side",
        "/placeholder.svg?height=500&width=500&text=Galaxy+S24+Back",
        "/placeholder.svg?height=500&width=500&text=S+Pen+Detail",
      ],
      rating: 4.8,
      reviews: 1923,
      badge: "New",
      category: "Smartphones",
      brand: "Samsung",
      description:
        "The Galaxy S24 Ultra combines cutting-edge AI features with the precision of the S Pen. Capture stunning photos with the 200MP camera and enjoy seamless productivity on the expansive display.",
      features: [
        "6.8-inch Dynamic AMOLED 2X display",
        "Snapdragon 8 Gen 3 processor",
        "200MP main camera with AI enhancement",
        "Built-in S Pen with Air Actions",
        "5000mAh battery with fast charging",
        "Galaxy AI features",
      ],
      specifications: {
        Display: "6.8-inch Dynamic AMOLED 2X",
        Processor: "Snapdragon 8 Gen 3",
        Storage: "256GB, 512GB, 1TB",
        Camera: "200MP Main, 50MP Periscope, 12MP Ultra Wide",
        Battery: "5000mAh with 45W fast charging",
        "Operating System": "Android 14 with One UI 6.1",
        Connectivity: "5G, Wi-Fi 7, Bluetooth 5.3",
        Colors: "Titanium Gray, Titanium Black, Titanium Violet, Titanium Yellow",
      },
      inStock: true,
      stockCount: 8,
    },
  }

  return products[id as keyof typeof products] || products["1"]
}

const relatedProducts = [
  {
    id: 3,
    name: "Google Pixel 8 Pro",
    price: 899,
    originalPrice: 999,
    image: "/placeholder.svg?height=200&width=200&text=Pixel+8+Pro",
    rating: 4.7,
    badge: "AI Powered",
  },
  {
    id: 4,
    name: "OnePlus 12",
    price: 799,
    originalPrice: 899,
    image: "/placeholder.svg?height=200&width=200&text=OnePlus+12",
    rating: 4.6,
    badge: "Fast Charging",
  },
  {
    id: 5,
    name: "iPhone 15",
    price: 799,
    originalPrice: 899,
    image: "/placeholder.svg?height=200&width=200&text=iPhone+15",
    rating: 4.8,
    badge: "Popular",
  },
  {
    id: 6,
    name: "Samsung Galaxy S24",
    price: 799,
    originalPrice: 899,
    image: "/placeholder.svg?height=200&width=200&text=Galaxy+S24",
    rating: 4.7,
    badge: "Compact",
  },
]

export default function ProductDetailPage({ params }: { params: { id: string } }) {
  const product = getProductById(params.id)
  const [selectedImage, setSelectedImage] = useState(0)
  const [quantity, setQuantity] = useState(1)
  const [selectedStorage, setSelectedStorage] = useState("256GB")

  const reviews = [
    {
      id: 1,
      name: "Sarah Johnson",
      rating: 5,
      date: "2 weeks ago",
      comment:
        "Absolutely love this phone! The camera quality is incredible and the battery lasts all day. Worth every penny.",
      verified: true,
    },
    {
      id: 2,
      name: "Mike Chen",
      rating: 5,
      date: "1 month ago",
      comment:
        "Upgraded from the previous model and the difference is night and day. The titanium build feels premium and the performance is blazing fast.",
      verified: true,
    },
    {
      id: 3,
      name: "Emily Rodriguez",
      rating: 4,
      date: "3 weeks ago",
      comment:
        "Great phone overall. The only minor complaint is that it's quite heavy, but the build quality makes up for it.",
      verified: true,
    },
  ]

  return (
    <div className="min-h-screen bg-background">
      {/* Navigation Header */}
      <header className="border-b bg-white/95 backdrop-blur-sm sticky top-0 z-50">
        <div className="container mx-auto px-4 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-8">
              <h1 className="text-2xl font-bold text-emerald-600">SavingVest Mart</h1>
              <nav className="hidden md:flex items-center gap-6">
                <a href="/" className="text-sm font-medium hover:text-emerald-600 transition-colors">
                  Home
                </a>
                <a href="/products" className="text-sm font-medium hover:text-emerald-600 transition-colors">
                  Products
                </a>
                <a href="#" className="text-sm font-medium hover:text-emerald-600 transition-colors">
                  Categories
                </a>
                <a href="#" className="text-sm font-medium hover:text-emerald-600 transition-colors">
                  Deals
                </a>
              </nav>
            </div>

            <div className="flex items-center gap-4">
              <div className="hidden md:flex items-center gap-2 bg-gray-100 rounded-full px-4 py-2 w-80">
                <Search className="h-4 w-4 text-gray-500" />
                <input
                  type="text"
                  placeholder="Search products..."
                  className="bg-transparent border-none outline-none flex-1 text-sm"
                />
              </div>
              <Button variant="ghost" size="icon">
                <User className="h-5 w-5" />
              </Button>
              <Button variant="ghost" size="icon" className="relative">
                <ShoppingCart className="h-5 w-5" />
                <Badge className="absolute -top-2 -right-2 h-5 w-5 rounded-full p-0 flex items-center justify-center text-xs">
                  3
                </Badge>
              </Button>
              <Button variant="ghost" size="icon" className="md:hidden">
                <Menu className="h-5 w-5" />
              </Button>
            </div>
          </div>
        </div>
      </header>

      {/* Breadcrumb */}
      <div className="bg-gray-50 py-4">
        <div className="container mx-auto px-4">
          <div className="flex items-center gap-2 text-sm text-gray-600">
            <a href="/" className="hover:text-emerald-600">
              Home
            </a>
            <ChevronRight className="h-4 w-4" />
            <a href="/products" className="hover:text-emerald-600">
              Products
            </a>
            <ChevronRight className="h-4 w-4" />
            <a href="/products" className="hover:text-emerald-600">
              {product.category}
            </a>
            <ChevronRight className="h-4 w-4" />
            <span className="text-gray-900">{product.name}</span>
          </div>
        </div>
      </div>

      <div className="container mx-auto px-4 py-8">
        <div className="grid lg:grid-cols-2 gap-12 mb-16">
          {/* Product Images */}
          <div className="space-y-4">
            <div className="relative overflow-hidden rounded-lg bg-gray-100">
              <img
                src={product.images[selectedImage] || "/placeholder.svg"}
                alt={product.name}
                className="w-full h-96 lg:h-[500px] object-cover"
              />
              <Badge className="absolute top-4 left-4 bg-emerald-600 hover:bg-emerald-600">{product.badge}</Badge>
            </div>

            <div className="grid grid-cols-4 gap-4">
              {product.images.map((image, index) => (
                <button
                  key={index}
                  onClick={() => setSelectedImage(index)}
                  className={`relative overflow-hidden rounded-lg border-2 transition-all ${
                    selectedImage === index ? "border-emerald-600" : "border-gray-200 hover:border-gray-300"
                  }`}
                >
                  <img
                    src={image || "/placeholder.svg"}
                    alt={`${product.name} view ${index + 1}`}
                    className="w-full h-20 object-cover"
                  />
                </button>
              ))}
            </div>
          </div>

          {/* Product Info */}
          <div className="space-y-6">
            <div>
              <div className="flex items-center gap-2 mb-2">
                <Badge variant="outline">{product.brand}</Badge>
                <Badge className="bg-emerald-100 text-emerald-700">{product.category}</Badge>
              </div>
              <h1 className="text-3xl lg:text-4xl font-bold text-gray-900 mb-4">{product.name}</h1>

              <div className="flex items-center gap-4 mb-4">
                <div className="flex items-center gap-2">
                  <div className="flex items-center gap-1">
                    {[...Array(5)].map((_, i) => (
                      <Star key={i} className="h-5 w-5 text-yellow-400 fill-current" />
                    ))}
                  </div>
                  <span className="font-medium">{product.rating}</span>
                  <span className="text-gray-600">({product.reviews} reviews)</span>
                </div>
              </div>

              <div className="flex items-center gap-4 mb-6">
                <span className="text-4xl font-bold text-gray-900">${product.price}</span>
                {product.originalPrice > product.price && (
                  <span className="text-2xl text-gray-500 line-through">${product.originalPrice}</span>
                )}
                {product.originalPrice > product.price && (
                  <Badge className="bg-red-100 text-red-700">Save ${product.originalPrice - product.price}</Badge>
                )}
              </div>

              <p className="text-lg text-gray-600 leading-relaxed mb-6">{product.description}</p>
            </div>

            {/* Storage Options */}
            <div className="space-y-4">
              <h3 className="font-semibold text-lg">Storage</h3>
              <div className="flex gap-3">
                {["256GB", "512GB", "1TB"].map((storage) => (
                  <Button
                    key={storage}
                    variant={selectedStorage === storage ? "default" : "outline"}
                    onClick={() => setSelectedStorage(storage)}
                    className={selectedStorage === storage ? "bg-emerald-600 hover:bg-emerald-700" : ""}
                  >
                    {storage}
                  </Button>
                ))}
              </div>
            </div>

            {/* Quantity and Add to Cart */}
            <div className="space-y-4">
              <div className="flex items-center gap-4">
                <div className="flex items-center gap-2">
                  <span className="font-medium">Quantity:</span>
                  <div className="flex items-center border rounded-lg">
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => setQuantity(Math.max(1, quantity - 1))}
                      disabled={quantity <= 1}
                    >
                      <Minus className="h-4 w-4" />
                    </Button>
                    <span className="px-4 py-2 font-medium">{quantity}</span>
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => setQuantity(Math.min(product.stockCount, quantity + 1))}
                      disabled={quantity >= product.stockCount}
                    >
                      <Plus className="h-4 w-4" />
                    </Button>
                  </div>
                </div>

                <div className="text-sm text-gray-600">
                  {product.inStock ? (
                    <span className="text-green-600">✓ In Stock ({product.stockCount} available)</span>
                  ) : (
                    <span className="text-red-600">Out of Stock</span>
                  )}
                </div>
              </div>

              <div className="flex gap-4">
                <Button
                  size="lg"
                  className="flex-1 bg-emerald-600 hover:bg-emerald-700 text-white"
                  disabled={!product.inStock}
                >
                  <ShoppingCart className="h-5 w-5 mr-2" />
                  Add to Cart
                </Button>
                <Button variant="outline" size="lg">
                  <Heart className="h-5 w-5" />
                </Button>
                <Button variant="outline" size="lg">
                  <Share2 className="h-5 w-5" />
                </Button>
              </div>
            </div>

            {/* Features */}
            <div className="space-y-4">
              <h3 className="font-semibold text-lg">Key Features</h3>
              <ul className="space-y-2">
                {product.features.map((feature, index) => (
                  <li key={index} className="flex items-center gap-2 text-gray-700">
                    <div className="w-2 h-2 bg-emerald-600 rounded-full flex-shrink-0"></div>
                    {feature}
                  </li>
                ))}
              </ul>
            </div>

            {/* Guarantees */}
            <div className="grid grid-cols-3 gap-4 pt-6 border-t">
              <div className="flex items-center gap-2 text-sm">
                <Truck className="h-5 w-5 text-emerald-600" />
                <span>Free Shipping</span>
              </div>
              <div className="flex items-center gap-2 text-sm">
                <Shield className="h-5 w-5 text-emerald-600" />
                <span>2-Year Warranty</span>
              </div>
              <div className="flex items-center gap-2 text-sm">
                <RotateCcw className="h-5 w-5 text-emerald-600" />
                <span>30-Day Returns</span>
              </div>
            </div>
          </div>
        </div>

        {/* Product Details Tabs */}
        <div className="mb-16">
          <Tabs defaultValue="specifications" className="w-full">
            <TabsList className="grid w-full grid-cols-3">
              <TabsTrigger value="specifications">Specifications</TabsTrigger>
              <TabsTrigger value="reviews">Reviews ({product.reviews})</TabsTrigger>
              <TabsTrigger value="shipping">Shipping & Returns</TabsTrigger>
            </TabsList>

            <TabsContent value="specifications" className="mt-6">
              <Card>
                <CardContent className="p-6">
                  <div className="grid md:grid-cols-2 gap-6">
                    {Object.entries(product.specifications).map(([key, value]) => (
                      <div key={key} className="flex justify-between py-2 border-b border-gray-100 last:border-b-0">
                        <span className="font-medium text-gray-900">{key}:</span>
                        <span className="text-gray-600">{value}</span>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="reviews" className="mt-6">
              <div className="space-y-6">
                <div className="flex items-center justify-between">
                  <h3 className="text-xl font-semibold">Customer Reviews</h3>
                  <Button variant="outline">Write a Review</Button>
                </div>

                <div className="space-y-6">
                  {reviews.map((review) => (
                    <Card key={review.id}>
                      <CardContent className="p-6">
                        <div className="flex items-start justify-between mb-4">
                          <div>
                            <div className="flex items-center gap-2 mb-1">
                              <span className="font-medium">{review.name}</span>
                              {review.verified && (
                                <Badge variant="outline" className="text-xs">
                                  Verified Purchase
                                </Badge>
                              )}
                            </div>
                            <div className="flex items-center gap-2">
                              <div className="flex items-center gap-1">
                                {[...Array(5)].map((_, i) => (
                                  <Star
                                    key={i}
                                    className={`h-4 w-4 ${i < review.rating ? "text-yellow-400 fill-current" : "text-gray-300"}`}
                                  />
                                ))}
                              </div>
                              <span className="text-sm text-gray-600">{review.date}</span>
                            </div>
                          </div>
                        </div>
                        <p className="text-gray-700">{review.comment}</p>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              </div>
            </TabsContent>

            <TabsContent value="shipping" className="mt-6">
              <Card>
                <CardContent className="p-6 space-y-6">
                  <div>
                    <h4 className="font-semibold mb-2">Shipping Information</h4>
                    <ul className="space-y-1 text-gray-600">
                      <li>• Free standard shipping on orders over $50</li>
                      <li>• Express shipping available for $9.99</li>
                      <li>• Same-day delivery available in select areas</li>
                      <li>• International shipping to 50+ countries</li>
                    </ul>
                  </div>

                  <div>
                    <h4 className="font-semibold mb-2">Return Policy</h4>
                    <ul className="space-y-1 text-gray-600">
                      <li>• 30-day return window for unused items</li>
                      <li>• Free return shipping on defective items</li>
                      <li>• Refunds processed within 5-7 business days</li>
                      <li>• Original packaging required for returns</li>
                    </ul>
                  </div>
                </CardContent>
              </Card>
            </TabsContent>
          </Tabs>
        </div>

        {/* Related Products */}
        <div>
          <h2 className="text-2xl font-bold text-gray-900 mb-8">Related Products</h2>
          <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-6">
            {relatedProducts.map((relatedProduct) => (
              <Card
                key={relatedProduct.id}
                className="group hover:shadow-xl transition-all duration-300 border-0 shadow-md"
              >
                <CardContent className="p-0">
                  <div className="relative overflow-hidden rounded-t-lg">
                    <img
                      src={relatedProduct.image || "/placeholder.svg"}
                      alt={relatedProduct.name}
                      className="w-full h-48 object-cover group-hover:scale-105 transition-transform duration-300"
                    />
                    <Badge className="absolute top-3 left-3 bg-emerald-600 hover:bg-emerald-600">
                      {relatedProduct.badge}
                    </Badge>
                  </div>

                  <div className="p-4 space-y-3">
                    <h3 className="font-semibold text-gray-900">{relatedProduct.name}</h3>
                    <div className="flex items-center gap-1 mb-2">
                      {[...Array(5)].map((_, i) => (
                        <Star key={i} className="h-4 w-4 text-yellow-400 fill-current" />
                      ))}
                      <span className="text-sm text-gray-600 ml-1">({relatedProduct.rating})</span>
                    </div>
                    <div className="flex items-center gap-2">
                      <span className="text-xl font-bold text-gray-900">${relatedProduct.price}</span>
                      <span className="text-sm text-gray-500 line-through">${relatedProduct.originalPrice}</span>
                    </div>
                    <Button className="w-full bg-emerald-600 hover:bg-emerald-700" size="sm">
                      View Details
                    </Button>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </div>

      {/* Footer */}
      <footer className="bg-gray-900 text-white py-16 mt-16">
        <div className="container mx-auto px-4">
          <div className="grid md:grid-cols-4 gap-8">
            <div className="space-y-4">
              <h3 className="text-2xl font-bold text-emerald-400">SavingVest Mart</h3>
              <p className="text-gray-400">Your trusted partner for premium products at unbeatable prices.</p>
            </div>

            <div className="space-y-4">
              <h4 className="font-semibold text-lg">Quick Links</h4>
              <div className="space-y-2">
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  About Us
                </a>
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Contact
                </a>
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Shipping Info
                </a>
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Returns
                </a>
              </div>
            </div>

            <div className="space-y-4">
              <h4 className="font-semibold text-lg">Categories</h4>
              <div className="space-y-2">
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Smartphones
                </a>
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Appliances
                </a>
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Electronics
                </a>
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Home & Garden
                </a>
              </div>
            </div>

            <div className="space-y-4">
              <h4 className="font-semibold text-lg">Support</h4>
              <div className="space-y-2">
                <p className="text-gray-400">📞 1-800-SAVINGS</p>
                <p className="text-gray-400">✉️ support@savingvest.com</p>
                <p className="text-gray-400">🕒 24/7 Customer Service</p>
              </div>
            </div>
          </div>

          <div className="border-t border-gray-800 mt-12 pt-8 text-center">
            <p className="text-gray-400">&copy; 2024 SavingVest Mart. All rights reserved.</p>
          </div>
        </div>
      </footer>
    </div>
  )
}
